"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.MultitenancyLifecycle = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _url = require("url");
var _lodash = require("lodash");
var _coreHttpRouterServerInternal = require("@kbn/core-http-router-server-internal");
var _multitenancy = require("../../../common/multitenancy");
/*
 *    Copyright 2020 floragunn GmbH
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

class MultitenancyLifecycle {
  constructor({
    authManager,
    kerberos,
    searchGuardBackend,
    configService,
    sessionStorageFactory,
    logger,
    pluginDependencies,
    spacesService,
    kibanaCore,
    clusterClient
  }) {
    (0, _defineProperty2.default)(this, "onPreAuth", async (request, response, toolkit) => {
      const authType = this.configService.get('searchguard.auth.type');
      const debugEnabled = this.configService.get('searchguard.multitenancy.debug');
      const externalTenant = this.getExternalTenant(request, debugEnabled);

      // If we have an externalTenant, we will continue so that we can
      // update the cookie's tenant value
      if (!this.isRelevantPath(request) && !externalTenant) {
        return toolkit.next();
      }
      let selectedTenant = null;
      const {
        authHeaders,
        sessionCookie
      } = await this.getSession(request);
      const isAuthenticatedRequest = authType === 'proxy' || authHeaders && authHeaders.authorization ? true : false;

      // We may run into ugly issues with the capabilities endpoint here if
      // we let through an unauthenticated request, despite try/catch
      // Hence, only call the tenant endpoint if we are using proxy
      // or have an authorization header.
      if (!isAuthenticatedRequest) {
        return toolkit.next();
      }

      // Check if MT is enabled in the backend
      const {
        kibana_mt_enabled
      } = await this.searchGuardBackend.getKibanaInfoWithInternalUser();
      this.configService.set('searchguard.multitenancy.enabled', kibana_mt_enabled || false);

      // Skip early if MT is not enabled
      if (!kibana_mt_enabled) {
        return toolkit.next();
      }
      try {
        if (this.clusterClient.config.requestHeadersWhitelist.indexOf('sgtenant') === -1) {
          this.clusterClient.config.requestHeadersWhitelist.push('sgtenant');
        }
      } catch (error) {
        this.logger.error(`Multitenancy: Could not check headers whitelist ${request.url.pathname}. ${error}`);
      }

      // The capabilities route may break the entire screen if
      // we get a 401 when retrieving the tenants. So for the
      // default capabilities, we can just skip MT here.
      if (request.url.pathname.indexOf('capabilities') > -1 && request.url.searchParams.get('useDefaultCapabilities') === "true") {
        return toolkit.next();
      }
      let userTenantInfo;
      try {
        // We need the user's data from the backend to validate the selected tenant
        userTenantInfo = await this.searchGuardBackend.getUserTenantInfo(authHeaders);
        if (!userTenantInfo.data.multi_tenancy_enabled) {
          // MT is disabled, we don't need to do anything.
          // This should have been checked earlier though, so this is just a fail safe.
          return toolkit.next();
        }
        selectedTenant = await this.getSelectedTenant({
          request,
          sessionCookie,
          username: userTenantInfo.data.username,
          externalTenant,
          userTenantInfo
        });
      } catch (error) {
        this.logger.error(`Multitenancy: Could not get tenant info from ${request.url.pathname}. ${error}`);
        if (error.statusCode === 401) {
          return toolkit.next();
        }
      }
      const requestHasRequestedTenant = externalTenant || typeof sessionCookie.tenant !== 'undefined';
      // If we have an external tenant, but the selectedTenant is null
      // after validation, that means that the user does not have
      // access to the requested tenant, or it does not exist
      if (selectedTenant === null && requestHasRequestedTenant) {
        if (request.url.pathname.startsWith('/app') || request.url.pathname === '/') {
          // If we have the wrong tenant in the cookie, we need to reset the cookie tenant value
          const shouldResetCookieTenant = !externalTenant && typeof sessionCookie.tenant !== 'undefined';
          if (shouldResetCookieTenant) {
            delete sessionCookie.tenant;
            await this.sessionStorageFactory.asScoped(request).set(sessionCookie);
          }
          if (request.url.searchParams.get('sgtenantsmenu') !== _multitenancy.MISSING_TENANT_PARAMETER_VALUE) {
            return response.redirected({
              body: 'Wrong tenant',
              //statusCode: 401,
              headers: {
                'location': this.basePath + `/app/home?sgtenantsmenu=` + _multitenancy.MISSING_TENANT_PARAMETER_VALUE
              }
            });
          }
        } else {
          // TODO maybe just pass through and let the backend return an error?
          return response.unauthorized();
        }
      }

      // We have a tenant to use
      if (selectedTenant !== null) {
        const rawRequest = (0, _coreHttpRouterServerInternal.ensureRawRequest)(request);
        (0, _lodash.assign)(rawRequest.headers, authHeaders, {
          sgtenant: selectedTenant
        });
        if (this.pluginDependencies.spaces) {
          // If we have a new tenant with no default space, we need to create it.
          await this.spacesService.createDefaultSpace({
            request,
            selectedTenant
          });
        }
        if (selectedTenant !== sessionCookie.tenant) {
          // save validated tenant in the cookie
          sessionCookie.tenant = selectedTenant;
          await this.sessionStorageFactory.asScoped(request).set(sessionCookie);
        }
      } else {
        let authRequiredForRoute = false;
        try {
          authRequiredForRoute = request.route.options.authRequired;
        } catch (error) {
          // Ignore
        }

        // Could also be "optional" or false
        if (authRequiredForRoute === true) {
          return response.redirected({
            body: 'Missing Tenant',
            //statusCode: 401,
            headers: {
              'location': this.basePath + `/searchguard/login?err=` + _multitenancy.MISSING_TENANT_PARAMETER_VALUE
            }
          });
        }
      }
      return toolkit.next();
    });
    /**
     * Get and validate the selected tenant.
     * @param request
     * @param sessionCookie
     * @param {string} username
     * @param {string|null} externalTenant
     * @param userTenantInfo
     * @returns {Promise<string|null>}
     */
    (0, _defineProperty2.default)(this, "getSelectedTenant", async ({
      request,
      sessionCookie,
      username,
      externalTenant,
      userTenantInfo
    }) => {
      const debugEnabled = this.configService.get('searchguard.multitenancy.debug');
      const backend = this.searchGuardBackend;

      // default is the tenant stored in the tenants cookie
      let selectedTenant = sessionCookie && typeof sessionCookie.tenant !== 'undefined' ? sessionCookie.tenant : null;
      if (debugEnabled) {
        this.logger.info(`Multitenancy: tenant_storagecookie ${selectedTenant}`);
      }
      if (externalTenant) {
        selectedTenant = externalTenant;
      }

      /**
       * @type {Record<string, boolean>}
       */
      const userTenants = this.searchGuardBackend.convertUserTenantsToRecord(userTenantInfo.data.tenants);

      // if we have a tenant, check validity and set it
      if (typeof selectedTenant !== 'undefined' && selectedTenant !== null && selectedTenant !== "") {
        selectedTenant = backend.validateRequestedTenant(username, selectedTenant, userTenants);
      } else if (userTenantInfo && userTenantInfo.data.default_tenant) {
        selectedTenant = userTenantInfo.data.default_tenant;
      }
      if (debugEnabled) {
        this.logger.info(`Multitenancy: tenant_assigned ${selectedTenant}`);
      }
      return selectedTenant;
    });
    /**
     * Get the auth information needed to make user scoped requests
     * @param request
     * @returns {Promise<{sessionCookie: {}, authHeaders: *}>}
     */
    (0, _defineProperty2.default)(this, "getSession", async request => {
      let sessionCookie;
      let authHeaders = request.headers;
      if (this.authManager) {
        const authInstance = await this.authManager.getAuthInstanceByRequest({
          request
        });
        if (authInstance) {
          sessionCookie = await authInstance.getCookieWithCredentials(request);
          authHeaders = authInstance.getAuthHeader(sessionCookie);
        } else {
          sessionCookie = await this.sessionStorageFactory.asScoped(request).get();
        }
      } else if (this.kerberos) {
        sessionCookie = await this.kerberos.getCookieWithCredentials(request);
        authHeaders = this.kerberos.getAuthHeader(sessionCookie);
      } else {
        sessionCookie = await this.sessionStorageFactory.asScoped(request).get();
      }
      if (!sessionCookie) {
        sessionCookie = {};
      }
      return {
        sessionCookie,
        authHeaders
      };
    });
    (0, _defineProperty2.default)(this, "isRelevantPath", request => {
      const path = request.url.pathname;

      // MT is only relevant for these paths
      const relevantPaths = ['/internal', '/goto', '/opensearch', '/app', '/api', '/bootstrap.js'];

      // MT is not relevant in these patterns
      const ignorePatterns = ['/api/status', '/api/v1/auth/config', '/api/v1/auth/login', '/api/v1/systeminfo'];
      return path === '/' || relevantPaths.some(root => path.startsWith(root)) && !ignorePatterns.some(root => path.startsWith(root));
    });
    /**
     * Check if we have a tenant set as query parameter
     * or as header value
     *
     * @param request
     * @param debugEnabled
     * @returns {null|string}
     */
    (0, _defineProperty2.default)(this, "getExternalTenant", (request, debugEnabled = false) => {
      let externalTenant = null;
      // check for tenant information in HTTP header. E.g. when using the saved objects API
      if (request.headers.sgtenant || request.headers.sg_tenant) {
        externalTenant = request.headers.sgtenant ? request.headers.sgtenant : request.headers.sg_tenant;
        if (debugEnabled) {
          this.logger.info(`Multitenancy: tenant_http_header: ${externalTenant}`);
        }
      }
      // check for tenant information in GET parameter. E.g. when using a share link. Overwrites the HTTP header.
      if (request.url.searchParams.has('sg_tenant') || request.url.searchParams.has('sgtenant')) {
        externalTenant = request.url.searchParams.has('sg_tenant') ? request.url.searchParams.get('sg_tenant') : request.url.searchParams.get('sgtenant');
        if (debugEnabled) {
          this.logger.info(`Multitenancy: tenant_url_param' ${externalTenant}`);
        }
      }
      if (externalTenant !== null) {
        try {
          if (externalTenant.toLowerCase() === 'private') {
            return _multitenancy.PRIVATE_TENANT_NAME;
          }
          if (externalTenant.toLowerCase() === 'global' || externalTenant.toUpperCase() === _multitenancy.GLOBAL_TENANT_NAME) {
            return _multitenancy.GLOBAL_TENANT_NAME;
          }
        } catch (error) {
          this.logger.error(`Could not translate private/global tenant: ` + externalTenant);
        }
      }
      return externalTenant;
    });
    this.authManager = authManager;
    this.searchGuardBackend = searchGuardBackend;
    this.configService = configService;
    this.sessionStorageFactory = sessionStorageFactory;
    this.logger = logger;
    this.pluginDependencies = pluginDependencies;
    this.spacesService = spacesService;
    this.kerberos = kerberos;
    this.kibanaCore = kibanaCore;
    this.clusterClient = clusterClient;
    this.basePath = kibanaCore.http.basePath.get();
  }
}
exports.MultitenancyLifecycle = MultitenancyLifecycle;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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